program LEASTSQUARESPOLYNOMIAL;
{--------------------------------------------------------------------}
{  Alg5'2.pas   Pascal program for implementing Algorithm 5.2        }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 5.2 (Least Squares Polynomial).                         }
{  Section   5.2, Curve Fitting, Page 278                            }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    MaxN = 15;
    MaxM = 100;
    GNmax = 630;

  type
    CVECTOR = array[1..MaxN] of real;
    DVECTOR = array[1..MaxM] of real;
    RVECTOR = array[0..GNmax] of real;
    MATRIX = array[1..MaxN, 1..MaxN] of real;
    LETTERS = string[200];
    Status = (Computing, Done, More, Working);
    DATYPE = (DatPoints, FunPoints);
    ABTYPE = (Given, Equal, Interval, Chebyshev);

  var
    DNpts, GNpts, Inum, M, N, Sub: integer;
    T, Rnum, Xmax, Xmin, Ymax, Ymin: real;
    X, Y: DVECTOR;
    B, C: CVECTOR;
    Xg, Yg: RVECTOR;
    A: MATRIX;
    Ans: CHAR;
    Mess: LETTERS;
    Stat, State: Status;
    Ytype: DATYPE;
    Xtype: ABTYPE;

  procedure FILLMATRIX (X, Y: DVECTOR; M: integer; var A: MATRIX; var B: CVECTOR; N: integer);
    var
      Col, J, K, R: integer;
      Pow: array[0..MaxN] of real;
      Prod, XK, YK: real;
  begin
    for R := 1 to N + 1 do                       {FILL COLUMN VECTOR}
      B[R] := 0;
    for K := 1 to M do
      begin
        YK := Y[K];
        XK := X[K];
        Prod := 1;
        for R := 1 to N + 1 do
          begin
            B[R] := B[R] + YK * Prod;
            Prod := Prod * XK;
          end;
      end;
    for J := 1 to 2 * N do                           {SUMS OF POWERS}
      Pow[J] := 0;
    Pow[0] := M;
    for K := 1 to M do
      begin
        XK := X[K];
        Prod := X[K];
        for J := 1 to 2 * N do
          begin
            Pow[J] := Pow[J] + Prod;
            Prod := Prod * XK;
          end;
      end;
    for R := 1 to N + 1 do                              {FILL MATRIX}
      begin
        for Col := 1 to N + 1 do
          begin
            A[R, Col] := Pow[R + Col - 2];
          end;
        WRITELN;
      end;
  end;                                         {END FILL PROCEDURE}


  procedure SolveLinSys (A: MATRIX; B: CVECTOR; N: integer; var C: CVECTOR);
    var
      Col, J, K, P, T: integer;
      Row: array[1..MaxN] of integer;
      Z: CVECTOR;
      Sum: real;
  begin
    for J := 1 to N + 1 do                {INITIALIZE POINTER VECTOR}
      Row[J] := J;
    for P := 1 to N do               {UPPER TRIANGULARIZATION LOOP}
      begin
        for K := P + 1 to N + 1 do
          begin                                        {FIND PIVOT ROW}
            if ABS(A[Row[K], P]) > ABS(A[Row[P], P]) then
              begin
                T := Row[P];
                Row[P] := Row[K];
                Row[K] := T;
              end;
          end;
        for K := P + 1 to N + 1 do               {GAUSSIAN ELIMINATION}
          begin
            A[Row[K], P] := A[Row[K], P] / A[Row[P], P];
            for Col := P + 1 to N + 1 do
              A[Row[K], Col] := A[Row[K], Col] - A[Row[K], P] * A[Row[P], Col];
          end;                             {END GAUSSIAN ELIMINATION}
      end;                              {END UPPER TRIANGULARIZATION}
    Z[1] := B[Row[1]];                       {FORWARD SUBSTITUTION}
    for K := 2 to N + 1 do
      begin
        Sum := 0;
        for Col := 1 to K - 1 do
          Sum := Sum + A[Row[K], Col] * Z[Col];
        Z[K] := B[Row[K]] - Sum;           {END FORWARD SUBSTITUTION}
      end;
    C[N + 1] := Z[N + 1] / A[Row[N + 1], N + 1];           {BACK SUBSTITUTION}
    for K := N downto 1 do
      begin
        Sum := 0;
        for Col := K + 1 to N + 1 do
          Sum := Sum + A[Row[K], Col] * C[Col];
        C[K] := (Z[K] - Sum) / A[Row[K], K];
      end;
  end;                                          {END LINEAR SYSTEM}

  function P (C: CVECTOR; N: integer; T: real): real;
    var
      K: integer;
      Sum: real;
  begin                                         {Synthetic division is}
    Sum := C[N + 1];
    for K := N downto 1 do                      {used to evaluate P(T)}
      begin
        Sum := C[K] + Sum * T;
      end;
    P := Sum;
  end;

  procedure PRINTPOLY (C: CVECTOR; N: integer);
    var
      K, U, V: integer;
  begin
    case N of
      0: 
        begin
          WRITELN;
          WRITELN('P(X)  =  C');
          WRITELN('          1');
          WRITELN;
        end;
      1: 
        begin
          WRITELN;
          WRITELN('P(X)  =  C X  +  C');
          WRITELN('          2       1');
          WRITELN;
        end;
      2: 
        begin
          WRITELN('            2');
          WRITELN('P(X)  =  C X   +  C X  +  C');
          WRITELN('          3        2       1');
          WRITELN;
        end;
      3: 
        begin
          WRITELN('            3        2');
          WRITELN('P(X)  =  C X   +  C X   +  C X  +  C ');
          WRITELN('          4        3        2       1');
          WRITELN;
        end;
      4, 5, 6, 7, 8: 
        begin
          WRITELN('            ', N, '        ', N - 1, '            2          ');
          WRITELN('P(X)  =  C X   +  C X   +...+  C X  +   C X  +  C ');
          WRITELN('          ', N + 1, '        ', N, '            3        2       1');
          WRITELN;
        end;
      9: 
        begin
          WRITELN('             ', N, '        ', N - 1, '            2          ');
          WRITELN('P(X)  =  C  X   +  C X   +...+  C X  +   C X  +  C ');
          WRITELN('          ', N + 1, '        ', N, '            3        2       1');
          WRITELN;
        end;
      else
        begin
          WRITELN('             ', N, '        ', N - 1, '            2          ');
          WRITELN('P(X)  =  C  X   +  C  X   +...+  C X  +   C X  +  C ');
          WRITELN('          ', N + 1, '        ', N, '            3        2       1');
          WRITELN;
        end;
    end;
    for K := 0 to TRUNC(N / 2) do                {Print the coefficients}
      begin
        U := 2 * K;                                {of P(X) in two columns}
        V := 2 * K + 1;
        if U <= N then
          begin
            begin
              if N < 9 then
                WRITE('C(', (N - U + 1) : 1, ')  =', C[N - U + 1] : 15 : 7, '         ')
              else
                WRITE('C(', (N - U + 1) : 2, ')  =', C[N - U + 1] : 15 : 7, '         ');
            end;
            if V <= N then
              begin
                if N < 9 then
                  WRITELN('C(', (N - V + 1) : 1, ')  =', C[N - V + 1] : 15 : 7)
                else
                  WRITELN('C(', (N - V + 1) : 2, ')  =', C[N - V + 1] : 15 : 7);
              end
            else
              WRITELN;
          end;
      end;
  end;

  procedure GETPOINTS (var X, Y: DVECTOR; var Xmin, Xmax: real; var M: integer; Stat: STATUS);
    type
      STATUS = (Bad, Enter, Done);
      LETTER = string[1];
    var
      Count, I, J, K, Kbad: integer;
      T, Valu: real;
      Resp: CHAR;
      Cond: STATUS;
  begin
    CLRSCR;
    Kbad := -1;
    State := Working;
    if Stat = More then
      begin
        for I := 1 to 6 do
          WRITELN;
        WRITE('Do you want  to enter new data points ?  <Y/N>  ');
        READLN(Resp);
        if (Resp = 'Y') or (Resp = 'y') then
          begin
            Stat := Working;
            CLRSCR;
          end;
      end;
    if (Stat = Working) then
      begin
        CLRSCR;
        Kbad := 0;
        for K := 1 to M do
          begin
            X[K] := 0;
            Y[K] := 0;
          end;
        CLRSCR;
        WRITELN;
        WRITELN('          Now enter the ', M : 2, ' points.');
        WRITELN;
        WRITELN('          You will have a chance to make changes at the end.');
        WRITELN;
        WRITELN;
        for K := 1 to M do
          begin
            X[K] := 0;
            Y[K] := 0;
          end;
        Xtype := Given;
        for K := 1 to M do
          begin
            if Xtype = Given then
              begin
                WRITELN;
                Mess := '         x';
                WRITE(Mess, K : 1, ' = ');
                READLN(X[K]);
              end;
            Ytype := DatPoints;
            if Ytype = DatPoints then
              begin
                if Xtype <> Given then
                  begin
                    WRITELN;
                    WRITELN('         x  =', X[K] : 15 : 7);
                    WRITE('          ', K : 0);
                  end;
                WRITELN;
                WRITELN;
                Mess := '         y';
                WRITE(Mess, K : 1, ' = ');
                READLN(Y[K]);
                WRITELN;
              end
            else
              begin
            {Y[K]:=F(X[K]); Provision for function values.}
              end;
            WRITELN;
          end;
      end;
    Xmin := X[1];
    Ymin := Y[1];
    for K := 1 to M do
      begin
        if (Xmin > X[K]) then
          Xmin := X[K];
        if (Ymin > Y[K]) then
          Ymin := Y[K];
      end;
    Cond := Enter;
    while (Cond = Enter) or (Cond = Bad) do
      begin
        CLRSCR;
        if (Cond = Bad) then
          WRITELN('     The abscissas are NOT distinct.   You MUST change one of them.');
        WRITELN('      k               x                     y');
        WRITELN('                       k                     k');
        WRITELN('----------------------------------------------------------------');
        for K := 1 to M do
          WRITELN('     ', K : 2, '       ', X[K] : 15 : 7, '       ', Y[K] : 15 : 7);
        WRITELN;
        if (Cond <> Bad) then
          begin
            WRITELN;
            if N > 15 then
              begin
                WRITELN;
              end;
            WRITE('     Are the points o.k. ?  <Y/N>  ');
            READLN(Resp);
          end;
        if (Resp = 'N') or (Resp = 'n') or (Cond = Bad) then
          begin
            if N > 14 then
              WRITELN;
            WRITELN;
            WRITELN;
            case M of
              2: 
                WRITELN('     To change a point select  k = 1,2');
              3: 
                WRITELN('     To change a point select  k = 1,2,3');
              else
                WRITELN('     To change a point select  k = 1,2,...,', M : 2);
            end;
            Mess := '                       ENTER   k = ';
            K := Kbad;
            WRITE(Mess);
            READLN(K);
            if (1 <= K) and (K <= M) then
              begin
                WRITELN;
                if K < 10 then
                  begin
                    WRITELN('     Coordinates of the  current point  (x ,y )  are:');
                    WRITELN('                                          ', K : 1, '  ', k : 1);
                    WRITELN('     Old   x  =', X[K] : 15 : 7, '      Old   y  =', Y[K] : 15 : 7);
                    WRITELN('            ', K : 1, '                              ', K : 1);
                  end
                else
                  begin
                    WRITELN('     Coordinates of the current point  (x  ,y  )  are:');
                    WRITELN('                                         ', K : 2, '  ', k : 2);
                    WRITELN('     Old   x  =', X[K] : 15 : 7, '      Old   y  =', Y[K] : 15 : 7);
                    WRITELN('            ', K : 2, '                             ', K : 2);
                  end;
                Mess := '     NEW   x';
                WRITE(Mess, K : 1, ' = ');
                READLN(X[K]);
                Mess := '     NEW   y';
                WRITE(Mess, K : 1, ' = ');
                READLN(Y[K]);
              end;
          end
        else
          Cond := Done;
        if (Cond = Bad) then
          Cond := Enter;
        for J := 1 to M - 1 do
          begin
            for K := J + 1 to M do
              if X[J] > X[K] then
                begin
                  T := X[J];
                  X[J] := X[K];
                  X[K] := T;
                  T := Y[J];
                  Y[J] := Y[K];
                  Y[K] := T;
                end;
          end;
        Kbad := 0;
        Count := 0;
        for K := 2 to M do
          if X[K] = X[K - 1] then
            begin
              Kbad := K;
              Count := Count + 1;
            end;
        if (M - Count <= N) then
          Cond := Bad;
      end;
    Xmax := X[1];
    Xmin := X[1];
    Ymax := Y[1];
    Ymin := Y[1];
    for K := 1 to M do
      begin
        if X[K] < Xmin then
          Xmin := X[K];
        if X[K] > Xmax then
          Xmax := X[K];
        if Y[K] < Ymin then
          Ymin := Y[K];
        if Y[K] > Ymax then
          Ymax := Y[K];
      end;
  end;

  procedure GETPOLY (var X, Y: DVECTOR; var M, N: integer);
    var
      K: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     The least squares polynomial  P (X)  of degree  M  is found');
    WRITELN('                                    M');
    WRITELN;
    WRITELN('which fits the  N  data points  (X ,Y ) , (X ,Y ) ,..., (X ,Y ).');
    WRITELN('                                  1  1      2  2          N  N');
   {Remark. The letter M and N are printed to match the text.}
   {        However the program uses the variable M for      }
   {        the number of data points and the variable       }
   {        N for the degree of the polynomial.              }
    Mess := '          ENTER the degree  M = ';   {See message above.}
    N := 0;
    WRITE(Mess);
    READLN(N);
    if N < 0 then
      N := 0;
    if N > 11 then
      N := 11;
    Mess := 'ENTER the number of points  N = ';   {See message above.}
    M := 0;
    WRITE(Mess);
    READLN(M);
    if M < N + 1 then
      begin
        WRITELN;
        WRITELN('You must have at least ', N + 1 : 2, ' points.');
        M := N + 1;
      end;
    WRITELN;
    GETPOINTS(X, Y, Xmin, Xmax, M, Stat);
  end;

  procedure RESULTS (C: CVECTOR; N: integer);
    var
      K: integer;
      Err, Z: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('The least squares polynomial is:');
    WRITELN;
    PRINTPOLY(C, N);
    WRITELN;
    WRITELN('    K       X             Y             P (X )           Error');
    WRITELN('             K             K             N  K');
    WRITELN('  -----------------------------------------------------------------');
    for K := 1 to M do
      begin
        Z := P(C, N, X[K]);
        Err := Y[K] - Z;
        WRITELN(K : 5, X[K] : 14 : 7, Y[K] : 14 : 7, '    ', Z : 14, '    ', Err : 12);
      end;
    WRITELN;
    WRITE('  Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
  end;

  procedure EVALUATE (C: CVECTOR; N: integer);
    var
      I: integer;
      T: real;
  begin
    CLRSCR;
    WRITELN;
    PRINTPOLY(C, N);
    WRITELN;
    WRITELN('Now evaluate  P(T)');
    Mess := 'ENTER a value   T = ';
    T := 0;
    WRITE(Mess);
    READLN(T);
    WRITELN;
    WRITELN;
    WRITELN('P(', T : 15 : 7, '  )  = ', P(C, N, T) : 15 : 7);
  end;

  procedure MESSAGE;
  begin
    CLRSCR;
    WRITELN('                        LEAST SQUARES POLYNOMIAL');
    WRITELN;
    WRITELN;
    WRITELN('          The least squares polynomial of degree M of the form:');
    WRITELN;
    WRITELN('                                          2               M');
    WRITELN('              P (x)  =  c   +  c x  +  c x  + ... +  c   x ');
    WRITELN('               M         1      2       3             M+1  ');
    WRITELN;
    WRITELN('     that fits the N data points (x ,y ), (x ,y ),..., (x ,y )');
    WRITELN('                                   1  1     2  2         N  N ');
    WRITELN;
    WRITELN('     will be found.  The coefficients  c , c ,...,c     will');
    WRITELN('                                        1   2      M+1');
    WRITELN;
    WRITELN('     minimize the error function:');
    WRITELN;
    WRITELN('                                    N ');
    WRITELN('                                                      2  ');
    WRITELN('             E(c ,c ,...,c   )  =  Sum ( P (x ) - y  ) . ');
    WRITELN('                1  2      M+1             M  k     k     ');
    WRITELN('                                   k=1');
    WRITELN;
    WRITELN;
    WRITE('                        Press the <ENTER> key. ');
    READLN(Ans);
    WRITELN;
  end;

  procedure MAKETABLE (var Xg, Yg: RVECTOR; GNpts: integer);
    var
      K0: integer;
      A0, B0: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     The polynomial curve fit will be plotted.');
    WRITELN;
    WRITELN('A portion of the curve  y = f(x)  will be computed and');
    WRITELN;
    WRITELN('graphed which lies over the the interval  a <= x <= b.');
    WRITELN;
    WRITELN;
    Mess := 'Enter the  left endpoint  a = ';
    A0 := X[1];
    WRITE(Mess);
    READLN(A0);
    WRITELN;
    Mess := 'Enter the right endpoint  b = ';
    B0 := X[M];
    WRITE(Mess);
    READLN(B0);
    CLRSCR;
    WRITELN;
    WRITELN('    In order to see the graphs, please make sure that');
    WRITELN;
    WRITELN('the computer has been set in the IBM color graphics mode.');
    WRITELN;
    WRITE('             Press the <ENTER> key.  ');
    READLN(Ans);
    WRITELN;
    WRITELN;
    WRITELN('             COMPUTING FUNCTION VALUES !');
    WRITELN;
    for K0 := 0 to GNpts do
      begin
        Xg[K0] := A0 + K0 * (B0 - A0) / GNpts;
        T := Xg[K0];
        Yg[K0] := P(C, N, T);
      end;
  end;

begin                                            {Begin Main Program}
  MESSAGE;
  Stat := Working;
  while Stat = Working do
    begin
      GETPOLY(X, Y, M, N);
      FILLMATRIX(X, Y, M, A, B, N);
      SolveLinSys(A, B, N, C);
      RESULTS(C, N);
      EVALUATE(C, N);
      WRITELN;
      WRITELN;
      WRITELN;
      WRITE('Want to use  a different set of  data ?  <Y/N>  ');
      READLN(Ans);
      WRITELN;
      if (Ans <> 'Y') and (Ans <> 'y') then
        Stat := Done;
    end;
end.                                            {End of Main Program}

